//	TorusGamesProjection.c
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesUtilities-Common.h"
#include "GeometryGamesMatrix44.h"


void MakeProjectionMatrix3D(
	ModelData	*md,						//	input
	double		aProjectionMatrix[4][4])	//	output
{
	unsigned int	i,
					j;
	
	//	In its default position, the frame cell has corners at (±1/2, ±1/2, ±1/2).
	//	We view it from the point (0,0,-1) with a 90° field of view:
	//
	//
	//			\                   /
	//			 \                 /
	//			  \   +-------+   /
	//			   \  |       |  /
	//			    \ |   o   | /
	//			     \|       |/
	//			      +-------+
	//			       \     /
	//			        \   /
	//			         \ /
	//				      *
	//
	//		o = origin (0,0,0)
	//		* = observer at (0,0,-1)
	//
	//	So we need a projection matrix with a ±45° field of view
	//	in each direction.  In other words, we want a view frustum
	//	with corners at the eight points
	//
	//			(±n, ±n, n, 1) and
	//			(±f, ±f, f, 1)
	//
	//	where n and f are the distances to the near and far
	//	clipping planes, respectively.  More precisely,
	//	because the GPU works in 4D homogeneous coordinates,
	//	it's really a set of eight rays, from the origin (0,0,0,0)
	//	through each of those eight points, that defines
	//	the view frustum as a "hyper-wedge" in 4D space.
	//
	//	Because the GPU works in homogenous coordinates,
	//	we may multiply each point by any scalar constant we like,
	//	without changing the ray that it represents.
	//	So let divide each of those eight points through
	//	by its own z coordinate, giving
	//
	//			(±1, ±1, 1, 1/n) and
	//			(±1, ±1, 1, 1/f)
	//
	//	Geometrically, these points define the intersection
	//	of the 4D wedge with the hyperplane z = 1.
	//	Conveniently enough, this intersection is a rectangular box!
	//
	//	Our goal is to find a 4×4 matrix that takes this rectangular box
	//	to the standard clipping box with corners at
	//
	//			(±1, ±1, 0, 1) and
	//			(±1, ±1, 1, 1)
	//
	//	To find such a matrix, let's "follow our nose"
	//	and construct it as the product of several factors.
	//	[Recall that, here and throughout the Geometry Games source code,
	//	matrices act using the left-to-right (row vector)(matrix) convention,
	//	not the right-to-left (matrix)(column vector) convention.]
	//
	//	Factor #1
	//
	//		The quarter turn matrix
	//
	//			1  0  0  0
	//			0  1  0  0
	//			0  0  0  1
	//			0  0 -1  0
	//
	//		takes the eight points
	//			(±1, ±1, 1, 1/n) and
	//			(±1, ±1, 1, 1/f)
	//		to
	//			(±1, ±1, -1/n, 1) and
	//			(±1, ±1, -1/f, 1)
	//
	//	Factor #2
	//
	//		The z dilation
	//
	//			1  0  0  0
	//			0  1  0  0
	//			0  0  a  0
	//			0  0  0  1
	//
	//		where a = n*f/(f - n), stretches or shrinks
	//		the box to have unit length in the z direction,
	//		taking
	//			(±1, ±1, -1/n, 1) and
	//			(±1, ±1, -1/f, 1)
	//		to
	//			(±1, ±1, -f/(f - n), 1) and
	//			(±1, ±1, -n/(f - n), 1)
	//
	//	Factor #3
	//
	//		The shear
	//
	//			1  0  0  0
	//			0  1  0  0
	//			0  0  1  0
	//			0  0  b  1
	//
	//		where b = f/(f - n), translates the hyperplane w = 1
	//		just the right amount to take
	//
	//			(±1, ±1, -f/(f - n), 1) and
	//			(±1, ±1, -n/(f - n), 1)
	//		to
	//			(±1, ±1, 0, 1) and
	//			(±1, ±1, 1, 1)
	//
	//		which are the vertices of the standard clipping box,
	//		which is exactly where we wanted to end up.
	//
	//	The projection matrix is the product (taken left-to-right!)
	//	of those three factors:
	//
	//			( 1  0  0  0 )( 1  0  0  0 )( 1  0  0  0 )
	//			( 0  1  0  0 )( 0  1  0  0 )( 0  1  0  0 )
	//			( 0  0  0  1 )( 0  0  a  0 )( 0  0  1  0 )
	//			( 0  0 -1  0 )( 0  0  0  1 )( 0  0  b  1 )
	//		=
	//			(  1   0   0   0 )
	//			(  0   1   0   0 )
	//			(  0   0   b   1 )
	//			(  0   0  -a   0 )
	//
	//	If we set the far clipping distance to f = ∞, the matrix simplifies to
	//
	//			(  1   0   0   0 )
	//			(  0   1   0   0 )
	//			(  0   0   1   1 )
	//			(  0   0  -n   0 )
	//
	//	To account for the offset between the observer (*) and the origin (o),
	//	premultiply by a 1-unit translation in the z-direction:
	//
	//		( 1  0  0  0 )( 1  0  0  0 )   ( 1  0  0  0 )
	//		( 0  1  0  0 )( 0  1  0  0 ) = ( 0  1  0  0 )
	//		( 0  0  1  0 )( 0  0  1  1 )   ( 0  0  1  1 )
	//		( 0  0  1  1 )( 0  0 -n  0 )   ( 0  0 1-n 1 )
	//
	Matrix44Identity(aProjectionMatrix);
	aProjectionMatrix[2][3] = 1.0;
	switch (md->itsViewType)
	{
		case ViewBasicLarge:
			//	The frame cell sits at some arbitrary orientation,
			//	so set the near clipping distance to
			//
			//		n = (origin-to-eye) - (origin-to-cube-corner)
			//		  = 1 - ½√3
			//
			//	to just avoid clipping a corner of the cube
			//	if it passes maximally close to the eye.
			aProjectionMatrix[3][2] = 0.5 * ROOT3;	//	1 - n = √3/2
			break;
		
		case ViewRepeating:
			//	Set the near clip plane to n = 1/2 to trim away
			//	any partial objects that may poke through
			//	the near face of the (axis-aligned) frame cell.
			aProjectionMatrix[3][2] = 0.5;			//	1 - n = 1/2
			break;
		
		default:
			GEOMETRY_GAMES_ABORT("SetProjectionMatrix() received an unexpected ViewType.");
			break;
	}
	
	//	Premultiply by a scaling during Simulation3DResetGameAsDomain….
	if (md->itsSimulationStatus == Simulation3DResetGameAsDomainPart1
	 || md->itsSimulationStatus == Simulation3DResetGameAsDomainPart2
	 || md->itsSimulationStatus == Simulation3DResetGameAsDomainPart3)
	{
		for (i = 0; i < 3; i++)		//	first three rows only
			for (j = 0; j < 4; j++)
				aProjectionMatrix[i][j] *= md->its3DResetGameScaleFactor;
	}
}
